/*$
aobook
Copyright (c) 2014-2025 Azel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
$*/

/*****************************************
 * ディレクトリ走査
 *****************************************/

#include <string.h>

#include <sys/types.h>
#include <dirent.h>

#include <mlk.h>
#include <mlk_dir.h>

#include <mlk_str.h>
#include <mlk_file.h>
#include <mlk_filestat.h>
#include <mlk_charset.h>


//-----------------

struct _mDir
{
	DIR *dir;
	struct dirent *curent;
	mStr strPath,	//検索パス
		strCurName;	//現在のファイル名 (UTF-8)
};

//-----------------


/**@ 閉じる */

void mDirClose(mDir *p)
{
	if(p)
	{
		mStrFree(&p->strPath);
		mStrFree(&p->strCurName);
	
		closedir(p->dir);

		mFree(p);
	}
}

/**@ ディレクトリを開く
 *
 * @d:パスがリンクの場合は、リンク先が開かれる。
 * @r:NULL でエラー */

mDir *mDirOpen(const char *path)
{
	mDir *p;
	char *str;
	DIR *dir;

	//開く

	str = mUTF8toLocale(path, -1, NULL);
	if(!str) return NULL;

	dir = opendir(str);

	mFree(str);

	if(!dir) return NULL;

	//確保

	p = (mDir *)mMalloc0(sizeof(mDir));
	if(!p)
	{
		closedir(dir);
		return NULL;
	}

	p->dir = dir;

	mStrSetText(&p->strPath, path);

	return p;
}

/**@ 次のエントリを取得
 *
 * @r:FALSE で終了 */

mlkbool mDirNext(mDir *p)
{
	struct dirent *ent;

	if(p)
	{
		ent = readdir(p->dir);
		if(ent)
		{
			p->curent = ent;
		
			mStrSetText_locale(&p->strCurName, ent->d_name, -1);
			return TRUE;
		}
	}
	
	return FALSE;
}

/**@ 現在のファイル名を取得
 *
 * @r:ポインタは解放しないこと */

const char *mDirGetFilename(mDir *p)
{
	return p->strCurName.buf;
}

/**@ 現在のファイル名を mStr に取得
 *
 * @p:fullpath TRUE で、フルパス名で取得 */

void mDirGetFilename_str(mDir *p,mStr *str,mlkbool fullpath)
{
	if(!p)
		mStrEmpty(str);
	else if(fullpath)
	{
		//フルパス
		mStrCopy(str, &p->strPath);
		mStrPathJoin(str, p->strCurName.buf);
	}
	else
		mStrCopy(str, &p->strCurName);
}

/**@ ファイル情報取得 */

mlkbool mDirGetStat(mDir *p,mFileStat *dst)
{
	mStr str = MSTR_INIT;
	mlkbool ret;

	mDirGetFilename_str(p, &str, TRUE);

	ret = mGetFileStat(str.buf, dst);

	mStrFree(&str);

	return ret;
}

/**@ 現在のファイルがディレクトリか
 *
 * @d:ファイル情報を取得して判定する。 */

mlkbool mDirIsDirectory(mDir *p)
{
	if(p && p->curent)
	{
		/* DIR 構造体の値で判定すると、
		 * リンクファイルであった場合は、リンク先の種類が判別できないので、
		 * ファイル情報を取得する。*/
		
		mFileStat st;

		if(mDirGetStat(p, &st))
			return ((st.flags & MFILESTAT_F_DIRECTORY) != 0);
	}

	return FALSE;
}

/**@ 現在のファイルが特殊な名前 ("." or "..") か */

mlkbool mDirIsSpecName(mDir *p)
{
	char *pc;

	if(p && p->strCurName.buf)
	{
		pc = p->strCurName.buf;
	
		if(*pc == '.')
			return (pc[1] == 0 || (pc[1] == '.' && pc[2] == 0));
	}

	return FALSE;
}

/**@ 現在のファイルが親へのリンク ("..") か */

mlkbool mDirIsSpecName_parent(mDir *p)
{
	return (p && p->strCurName.buf && strcmp(p->strCurName.buf, "..") == 0);
}

/**@ 現在のファイルが隠しファイルか
 *
 * @d:先頭が '.' なら隠しファイル。 */

mlkbool mDirIsHiddenFile(mDir *p)
{
	return (p && p->strCurName.buf && p->strCurName.buf[0] == '.');
}

/**@ 現在のファイルの拡張子を比較 (大/小文字区別しない) */

mlkbool mDirCompareExt(mDir *p,const char *ext)
{
	return (p && mStrPathCompareExtEq(&p->strCurName, ext));
}
