/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2021 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#include <dirent.h>
#include <limits.h>
#include <stdlib.h>

#include "flow.h"
#include "io/dir.h"
#include "mem.h"
#include "util.h"

struct DirIterator {
	DIR *dir;
	int forwarded;
	size_t i;
	size_t len;
};

struct DirIterator *
dir_iterator(DIR *dir, ssize_t a, ssize_t b)
{
	struct DirIterator *iter = xmalloc(sizeof(struct DirIterator));
	slice_to_range(SSIZE_MAX, a, b, &iter->i, &iter->len);
	iter->dir = dir;
	return iter;
}

void
dir_iterator_cleanup(struct DirIterator **iter_)
{
	struct DirIterator *iter = *iter_;
	if (iter) {
		free(iter);
		*iter_ = NULL;
	}
}

int
dir_iterator_next(struct DirIterator **iter_, size_t *index, struct dirent **key, struct dirent **val)
{
	struct DirIterator *iter = *iter_;

	unless (iter->forwarded) {
		iter->forwarded = 1;
		for (size_t i = 0; i < iter->i; i++) {
			unless (readdir(iter->dir)) {
				goto done;
			}
		}
	}

	if (iter->i < iter->len) {
		struct dirent *ent = readdir(iter->dir);
		unless (ent) {
			goto done;
		}
		*index = iter->i++;
		*key = ent;
		*val = ent;
		return 1;
	}

done:
	dir_iterator_cleanup(iter_);
	*iter_ = NULL;
	return 0;
}
