/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2021 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#include <stdlib.h>
#include <unistd.h>

#include "array.h"
#include "flow.h"
#include "map.h"
#include "mem.h"
#include "mempool.h"
#include "queue.h"
#include "set.h"
#include "stack.h"

struct Mempool {
	struct Stack *stack;
};

struct MempoolNode {
	void *ptr;
	void (*freefn)(void *);
};

struct Mempool *
mempool_new()
{
	struct Mempool *pool = xmalloc(sizeof(struct Mempool));
	pool->stack = stack_new();
	return pool;
}

void
mempool_free(struct Mempool *pool)
{
	unless (pool) {
		return;
	}
	mempool_release_all(pool);
	stack_free(pool->stack);
	free(pool);
}

void
mempool_cleanup(struct Mempool **pool)
{
	if (pool) {
		mempool_free(*pool);
		*pool = NULL;
	}
}

void *
mempool_add(struct Mempool *pool, void *ptr, void *freefn)
{
	if (!pool || !ptr || !freefn) {
		return ptr;
	}

	struct MempoolNode *node = xmalloc(sizeof(struct MempoolNode));
	node->ptr = ptr;
	node->freefn = freefn;
	stack_push(pool->stack, node);

	return ptr;
}

void *
mempool_forget(struct Mempool *pool, void *ptr)
{
	if (!pool) {
		return ptr;
	}

	STACK_FOREACH(pool->stack, struct MempoolNode *, node) {
		if (node->ptr == ptr) {
			node->ptr = NULL;
			break;
		}
	}

	return ptr;
}

void
mempool_inherit(struct Mempool *pool, struct Mempool *other)
{
	struct MempoolNode *node;
	while ((node = stack_pop(other->stack))) {
		if (node->ptr) {
			stack_push(pool->stack, node);
		} else {
			free(node);
		}
	}
}

void *
mempool_move(struct Mempool *pool, void *ptr, struct Mempool *other)
{
	if (!pool || pool == other || !ptr) {
		return ptr;
	}
	STACK_FOREACH(pool->stack, struct MempoolNode *, node) {
		if (node->ptr == ptr) {
			node->ptr = NULL;
			if (other) {
				mempool_add(other, ptr, node->freefn);
			}
			break;
		}
	}
	return ptr;
}

void *
mempool_release(struct Mempool *pool, void *ptr)
{
	if (!pool) {
		return ptr;
	}

	STACK_FOREACH(pool->stack, struct MempoolNode *, node) {
		if (node->ptr == ptr) {
			node->freefn(ptr);
			node->ptr = NULL;
			break;
		}
	}

	return ptr;
}

void
mempool_release_all(struct Mempool *pool)
{
	if (!pool) {
		return;
	}

	struct MempoolNode *node;
	while ((node = stack_pop(pool->stack))) {
		if (node->ptr) {
			node->freefn(node->ptr);
		}
		free(node);
	}
}

void *
mempool_alloc(struct Mempool *pool, size_t sz)
{
	return mempool_add(pool, xmalloc(sz), free);
}

void *
mempool_take(struct Mempool *pool, void *ptr)
{
	return mempool_add(pool, ptr, free);
}
