/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2021 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "flow.h"
#include "mempool.h"
#include "path.h"
#include "str.h"

// Prototypes
static void path_normalize_helper(struct Mempool *, const char *, struct Array *);

char *
path_join(struct Mempool *extpool, struct Array *components)
{
	SCOPE_MEMPOOL(pool);
	return path_normalize(extpool, str_join(pool, components, "/"), NULL);
}

void
path_normalize_helper(struct Mempool *pool, const char *path, struct Array *components)
{
	size_t dirs = array_len(components);
	ARRAY_FOREACH(str_split(pool, path, "/"), const char *, component) {
		if (*component) {
			if (strcmp(component, ".") == 0) {
				// skip
			} else if (strcmp(component, "..") == 0) {
				if (dirs > 0) {
					unless (array_len(components) == 1 && strcmp(array_get(components, 0), "") == 0) {
						dirs--;
						array_pop(components);
					}
				} else {
					array_append(components, component);
				}
			} else {
				dirs++;
				array_append(components, component);
			}
		} else if (component_index == 0) {
			dirs++;
			array_append(components, "");
		}
	}
}

char *
path_normalize(struct Mempool *extpool, const char *path, const char *cwd)
{
	SCOPE_MEMPOOL(pool);

	panic_unless(path, "path is NULL");

	struct Array *components = mempool_array(pool);
	if (cwd) {
		path_normalize_helper(pool, cwd, components);
	}
	if (*path) {
		path_normalize_helper(pool, path, components);
	}

	return str_join(extpool, components, "/");
}

struct Array *
path_split(struct Mempool *pool, const char *origpath)
{
	return str_split(pool, path_normalize(pool, origpath, NULL), "/");
}
