/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2021 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "config.h"

#include <stdlib.h>

#include "peg.h"
#include "peg/distinfo.h"
#include "peg/grammar.h"

// Prototypes
static RULE(checksum);
static RULE(checksum_tail);
static RULE(digit);
static RULE(hexdigit);
static RULE(separator);
static RULE(size);
static RULE(size_or_checksum);
static RULE(size_tail);
static RULE(space);
static RULE(timestamp);

RULE(digit) {
	if (RANGE(0x30, 0x39)) // 0-9
	return ACCEPT;
	return ERROR("expected digit");
}

RULE(hexdigit) {
	if (!RANGE(0x30, 0x39)) // 0-9
	if (!RANGE(0x41, 0x46)) // A-F
	if (!RANGE(0x61, 0x66)) // a-f
	return ERROR("expected hexdigit");
	return ACCEPT;
}

RULE(space) {
	if (CHAR(' ', '\t'))
	return ACCEPT;
	return ERROR("expected space or tab");
}

RULE(timestamp) {
	if (STRING("TIMESTAMP"))
	if (ANY(space))
	if (CHAR('='))
	if (ANY(space))
	if (CAPTURE(SOME(digit), 0, PEG_DISTINFO_TIMESTAMP))
	if (CHAR('\n'))
	return ACCEPT;
	return REJECT;
}

RULE(separator) {
	if (CHAR(')'))
	if (ANY(space))
	if (CHAR('='))
	if (ANY(space))
	return ACCEPT;
	return REJECT;
}

RULE(checksum_tail) {
	if (MATCH(separator))
	if (CAPTURE(REPEAT(hexdigit, 64), PEG_DISTINFO_CHECKSUM_ALGORITHM, PEG_DISTINFO_DIGEST))
	if (CHAR('\n'))
	return ACCEPT;
	return REJECT;
}

RULE(checksum) {
	if (CAPTURE(STRING("SHA256"), 0, PEG_DISTINFO_CHECKSUM_ALGORITHM))
	if (ANY(space))
	if (CHAR('('))
	if (CAPTURE(MATCH_TO(checksum_tail), PEG_DISTINFO_CHECKSUM_ALGORITHM, PEG_DISTINFO_FILENAME))
	if (MATCH(checksum_tail))
	return ACCEPT;
	return ERROR("expected SHA256");
}

RULE(size_tail) {
	if (MATCH(separator))
	if (CAPTURE(SOME(digit), PEG_DISTINFO_SIZE, PEG_DISTINFO_SIZE))
	if (CHAR('\n'))
	return ACCEPT;
	return REJECT;
}

RULE(size) {
	if (STRING("SIZE"))
	if (ANY(space))
	if (CHAR('('))
	if (CAPTURE(MATCH_TO(size_tail), PEG_DISTINFO_SIZE, PEG_DISTINFO_FILENAME))
	if (MATCH(size_tail))
	return ACCEPT;
	return ERROR("expected SIZE");
}

RULE(size_or_checksum) {
	if (!MATCH(size))
	if (!MATCH(checksum))
	return ERROR("expected SHA256 or SIZE");
	return ACCEPT;
}

RULE(peg_distinfo_decode) {
	if (OPT(MATCH(timestamp)))
	if (ANY(size_or_checksum))
	if (EOS())
	return ACCEPT;
	return REJECT;
}
