/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2020 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#include <stdlib.h>

#include "array.h"
#include "map.h"
#include "mem.h"
#include "set.h"

struct Set {
	struct Map *map;
};

struct SetIterator {
	struct MapIterator *iter;
};

struct Set *
set_new(struct CompareTrait *compare)
{
	struct Set *set = xmalloc(sizeof(struct Set));
	set->map = map_new(compare);
	return set;
}

void
set_free(struct Set *set)
{
	if (set == NULL) {
		return;
	}

	map_free(set->map);
	free(set);
}

void
set_add(struct Set *set, const void *element)
{
	map_add(set->map, element, element);
}

void
set_remove(struct Set *set, const void *element)
{
	map_remove(set->map, element);
}

void *
set_get(struct Set *set, const void *element)
{
	return map_get(set->map, element);
}

bool
set_contains(struct Set *set, const void *element)
{
	return map_get(set->map, element) != NULL;
}

size_t
set_len(struct Set *set)
{
	return map_len(set->map);
}

void
set_truncate(struct Set *set)
{
	map_truncate(set->map);
}

struct Array *
set_values(struct Set *set, struct Mempool *pool)
{
	return map_values(set->map, pool);
}

struct SetIterator *
set_iterator(struct Set *set, ssize_t a, ssize_t b)
{
	struct SetIterator *iter = xmalloc(sizeof(struct SetIterator));
	iter->iter = map_iterator(set->map, a, b);
	return iter;
}

void
set_iterator_cleanup(struct SetIterator **iter_)
{
	struct SetIterator *iter = *iter_;
	if (iter != NULL) {
		map_iterator_cleanup(&iter->iter);
		free(iter);
		*iter_ = NULL;
	}
}

bool
set_iterator_next(struct SetIterator **iter_, size_t *index, void **key, void **value)
{
	struct SetIterator *iter = *iter_;
	if (map_iterator_next(&iter->iter, index, key, value)) {
		return true;
	} else {
		set_iterator_cleanup(iter_);
		*iter_ = NULL;
		return false;
	}
}
