/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2021 Tobias Kortkamp <tobik@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#include <sys/param.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "flow.h"
#include "mempool.h"
#include "str.h"
#include "trait/compare.h"
#include "util.h"

// Prototypes
static DECLARE_COMPARE(compare_str);
static DECLARE_COMPARE(casecompare_str);
static char *xstrdup(const char *);
static char *xstrndup(const char *, size_t);

// Constants
struct CompareTrait *str_compare = &(struct CompareTrait){
	.compare = compare_str,
	.compare_userdata = NULL,
};
struct CompareTrait *str_casecompare = &(struct CompareTrait){
	.compare = casecompare_str,
	.compare_userdata = NULL,
};

DEFINE_COMPARE(casecompare_str, const char, void)
{
	return strcasecmp(a, b);
}

DEFINE_COMPARE(compare_str, const char, void)
{
	return strcmp(a, b);
}

char *
str_common_prefix(struct Mempool *pool, const char *a, const char *b)
{
	const char *ap = a;
	const char *bp = b;
	size_t i;
	for (i = 0; *ap != 0 && *bp != 0 && *ap++ == *bp++; i++);
	if (i > 0) {
		return mempool_take(pool, xstrndup(a, i));
	} else {
		return NULL;
	}
}

char *
str_dup(struct Mempool *pool, const char *s)
{
	return mempool_take(pool, xstrdup(s));
}

char *
str_ndup(struct Mempool *pool, const char *s, const size_t n)
{
	return mempool_take(pool, xstrndup(s, n));
}

bool
str_endswith(const char *s, const char *end)
{
	size_t len = strlen(end);
	if (strlen(s) < len) {
		return false;
	}
	return strncmp(s + strlen(s) - len, end, len) == 0;
}

char *
str_join(struct Mempool *pool, struct Array *array, const char *sep)
{
	const size_t seplen = strlen(sep);
	const size_t lastindex = array_len(array) - 1;

	size_t sz = 1;
	ARRAY_FOREACH(array, const char *, s) {
		sz += strlen(s);
		if (s_index != lastindex) {
			sz += seplen;
		}
	}

	char *buf = mempool_alloc(pool, sz);
	char *p = buf;
	ARRAY_FOREACH(array, const char *, s) {
		const size_t len = strlen(s);
		memcpy(p, s, len);
		p += len;
		if (s_index != lastindex) {
			memcpy(p, sep, seplen);
			p += seplen;
		}
	}

	return buf;
}

char *
str_map(struct Mempool *pool, const char *s, size_t len, int (*f)(int))
{
	char *buf = mempool_alloc(pool, len + 1);
	for (size_t i = 0; i < len && s[i] != 0; i++) {
		buf[i] = f((unsigned char)s[i]);
	}
	return buf;
}

char *
str_printf(struct Mempool *pool, const char *format, ...)
{
	va_list ap;
	va_start(ap, format);
	char *buf = NULL;
	panic_if(vasprintf(&buf, format, ap) < 0, "vasprintf");
	mempool_take(pool, buf);
	va_end(ap);

	return buf;
}

char *
str_repeat(struct Mempool *pool, const char *s, const size_t n)
{
	const size_t len = strlen(s);
	const size_t sz = len * n + 1;
	char *buf = mempool_alloc(pool, sz);
	if (n > 0) {
		char *p = buf;
		for (size_t i = 0; i < n; i++) {
			memcpy(p, s, len);
			p += len;
		}
	}
	return buf;
}

char *
str_slice(struct Mempool *pool, const char *s, const ssize_t a, const ssize_t b)
{
	size_t len = strlen(s);
	size_t start = 0;
	size_t end = 0;
	slice_to_range(len, a, b, &start, &end);
	return str_ndup(pool, s + start, end - start);
}

struct Array *
str_split(struct Mempool *pool, const char *s, const char *sep)
{
	return str_nsplit(pool, s, strlen(s), sep);
}

struct Array *
str_nsplit(struct Mempool *pool, const char *s, const size_t slen, const char *sep)
{
	struct Array *array = mempool_array(pool);
	size_t seplen = strlen(sep);
	const char *buf = str_ndup(pool, s, slen);
	size_t buflen = slen;
	if (seplen > 0) {
		char *ptr;
		while (*buf && (ptr = strnstr(buf, sep, buflen))) {
			*ptr = 0;
			array_append(array, buf);
			buf = ptr + seplen;
			buflen -= seplen;
		}
	}
	array_append(array, buf);
	return array;
}

bool
str_startswith(const char *s, const char *start)
{
	size_t len = strlen(start);
	if (strlen(s) < len) {
		return false;
	}
	return strncmp(s, start, len) == 0;
}

char *
str_trim(struct Mempool *pool, const char *s)
{
	const char *sp = s;
	for (; *sp && isspace((unsigned char)*sp); ++sp);
	return str_trimr(pool, sp);
}

char *
str_triml(struct Mempool *pool, const char *s)
{
	const char *sp = s;
	for (; *sp && isspace((unsigned char)*sp); ++sp);
	return mempool_take(pool, xstrdup(sp));
}

char *
str_trimr(struct Mempool *pool, const char *s)
{
	size_t len = strlen(s);
	while (len > 0 && isspace((unsigned char)s[len - 1])) {
		len--;
	}
	return mempool_take(pool, xstrndup(s, len));
}

char *
xstrdup(const char *s)
{
	char *retval = strdup(s);
	panic_unless(retval, "strdup");
	return retval;
}

char *
xstrndup(const char *s, size_t len)
{
	char *retval = strndup(s, len);
	panic_unless(retval, "strndup");
	return retval;
}
